<?php

namespace App\Http\Controllers;

use App\Models\Post;
use App\Models\Category;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    // 1. Homepage Logic
  public function index()
    {
        // 1. Featured Jobs (Sirf wahi jo Admin ne Highlight ki hain)
        $data['featured_jobs'] = Post::where('post_type', 'job')
                                ->where('status', 1)
                                ->where('is_featured', 1) // Ye line magic karegi
                                ->latest()
                                ->take(15) // Top 15 featured posts
                                ->get();

        // 2. Admit Cards (Normal)
        $data['admit_cards'] = Post::where('post_type', 'admit_card')
                                ->where('status', 1)
                                ->latest()
                                ->take(10)
                                ->get();

        // 3. Results (Normal)
        $data['results'] = Post::where('post_type', 'result')
                                ->where('status', 1)
                                ->latest()
                                ->take(10)
                                ->get();
        
        // 4. Marquee (Top Scrolling Line)
        $data['marquee_posts'] = Post::where('is_marquee', 1)
                                ->where('status', 1)
                                ->latest()
                                ->get();

        return view('home', $data);
    }
    // 4. View All / Archive Page Logic
    public function viewAll($type)
    {
        // URL se 'type' milega (job, admit_card, result etc.)
        // Hum check karenge ki ye valid hai ya nahi
        $validTypes = ['job', 'admit_card', 'result', 'answer_key', 'syllabus', 'admission'];
        
        if (!in_array($type, $validTypes)) {
            abort(404); // Agar galat type dala toh 404 Error
        }

        // Title set karna display ke liye
        $pageTitle = match ($type) {
            'job' => 'Latest Government Jobs',
            'admit_card' => 'Download Admit Cards',
            'result' => 'Exam Results',
            'answer_key' => 'Answer Keys',
            'syllabus' => 'Syllabus',
            'admission' => 'Admission Forms',
            default => 'All Posts',
        };

        // Database se data fetch karna (Pagination ke saath - 20 per page)
        $posts = Post::where('post_type', $type)
                    ->where('status', 1)
                    ->latest() // Newest first
                    ->paginate(20); // Ek page par 20 posts

        return view('view-all', compact('posts', 'pageTitle', 'type'));
    }
   
    // 2. Single Post Page (Detail Page)
    public function show($slug)
    {
        $post = Post::where('slug', $slug)->where('status', 1)->firstOrFail();
        
        // Agar post "Job" se "Admit Card" ban chuki hai, tab bhi isi page par dikhegi
        return view('post-detail', compact('post'));
    }

    // 3. Category Page (View All)
    public function category($slug)
    {
        // Logic baad me add karenge agar "View All" page chahiye
    }

     // 5. Search Functionality
    public function search(Request $request)
    {
        $query = $request->input('query');

        // Agar search box khali hai toh wapas home bhej do
        if (!$query) {
            return redirect()->route('home');
        }

        // Database Search Query
        $posts = Post::where('status', 1) // Sirf Active Posts
            ->where(function($q) use ($query) {
                $q->where('title', 'LIKE', "%{$query}%")     // Title match kare
                  ->orWhere('post_name', 'LIKE', "%{$query}%") // Exam Name match kare
                  ->orWhere('meta_keywords', 'LIKE', "%{$query}%"); // Keywords match kare
            })
            ->latest()
            ->paginate(20); // 20 results per page

        return view('search-results', compact('posts', 'query'));
    }

    public function ajaxSearch(Request $request)
    {
        $query = $request->get('query');

        if (!$query) {
            return response()->json([]);
        }

        $posts = Post::where('status', 1)
            ->where(function($q) use ($query) {
                // Is group ke andar OR chalega, bahar AND rahega
                $q->where('title', 'LIKE', "%{$query}%")
                  ->orWhere('post_name', 'LIKE', "%{$query}%");
            })
            ->select('title', 'slug', 'post_type')
            ->latest()
            ->limit(6)
            ->get();

        return response()->json($posts);
    }
    // 6. Static Page Viewer
    public function page($slug)
    {
        $page = \App\Models\Page::where('slug', $slug)->where('status', 1)->firstOrFail();
        return view('page', compact('page'));
    }

}